# This document is part of lyx2lyx
# Copyright (C) 2002-2004 José Matos <jamatos@lyx.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

"""Convert files to the file format generated by lyx 1.1.5"""

import re

from parser_tools import find_re, find_token, find_token_backwards

####################################################################
# Private helper functions


def get_layout(line, default_layout):
    "Get the line layout, beware of the empty layout."
    tokens = line.split()
    if len(tokens) > 1:
        return tokens[1]
    return default_layout


####################################################################

math_env = ["\\[", "\\begin{eqnarray*}", "\\begin{eqnarray}", "\\begin{equation}"]


def replace_protected_separator(document):
    "Replace protected separator."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\protected_separator", i)
        if i == -1:
            break
        j = find_token_backwards(lines, "\\layout", i)
        # if j == -1: print error
        layout = get_layout(lines[j], document.default_layout)

        if layout == "LyX-Code":
            result = ""
            while lines[i] == "\\protected_separator ":
                result = result + " "
                del lines[i]

            lines[i - 1] = lines[i - 1] + result + lines[i]
        else:
            lines[i - 1] = lines[i - 1] + "\\SpecialChar ~"

        del lines[i]


def merge_formula_inset(document):
    "Merge formula insets."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\begin_inset Formula", i)
        if i == -1:
            break
        if lines[i + 1] in math_env:
            lines[i] = lines[i] + lines[i + 1]
            del lines[i + 1]
        i = i + 1


def update_tabular(document):
    "Update from tabular format 4 to 5 if necessary."
    lines = document.body
    lyxtable_re = re.compile(r".*\\LyXTable$")
    i = 0
    while True:
        i = find_re(lines, lyxtable_re, i)
        if i == -1:
            break
        i = i + 1
        format = lines[i][8]
        if format != "4":
            continue

        lines[i] = "multicol5"
        i = i + 1
        rows = int(lines[i].split()[0])
        columns = int(lines[i].split()[1])

        i = i + rows + 1
        for j in range(columns):
            col_info = lines[i].split()
            if len(col_info) == 3:
                lines[i] = lines[i] + '"" ""'
            else:
                lines[i] = " ".join(col_info[:3]) + ' "%s" ""' % col_info[3]
            i = i + 1

        while lines[i]:
            lines[i] = lines[i] + ' "" ""'
            i = i + 1


def update_toc(document):
    "Update table of contents."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\begin_inset LatexCommand \\tableofcontents", i)
        if i == -1:
            break
        lines[i] = lines[i] + "{}"
        i = i + 1


def remove_cursor(document):
    "Remove cursor."
    lines = document.body
    i = find_token(lines, "\\cursor", 0)
    if i != -1:
        del lines[i]


def remove_vcid(document):
    "Remove \\lyxvcid and \\lyxrcsid."
    lines = document.header
    i = find_token(lines, "\\lyxvcid", 0)
    if i != -1:
        del lines[i]
    i = find_token(lines, "\\lyxrcsid", 0)
    if i != -1:
        del lines[i]


def first_layout(document):
    "Fix first layout, if empty use the default layout."
    lines = document.body
    while lines[0] == "":
        del lines[0]
    if lines[0][:7] != "\\layout":
        lines[:0] = ["\\layout %s" % document.default_layout, ""]


def remove_space_in_units(document):
    "Remove space in units."
    lines = document.header
    margins = ["\\topmargin", "\\rightmargin", "\\leftmargin", "\\bottommargin"]

    unit_rexp = re.compile(r"[^ ]* (.*) (.*)")

    for margin in margins:
        i = 0
        while True:
            i = find_token(lines, margin, i)
            if i == -1:
                break

            result = unit_rexp.search(lines[i])
            if result:
                lines[i] = margin + " " + result.group(1) + result.group(2)
            i = i + 1


def latexdel_getargs(document, i):
    "Get arguments from latexdel insets."
    lines = document.body

    # play safe, clean empty lines
    while True:
        if lines[i]:
            break
        del lines[i]

    j = find_token(lines, "\\end_inset", i)

    if i == j:
        del lines[i]
    else:
        document.warning("Unexpected end of inset.")
    j = find_token(lines, "\\begin_inset LatexDel }{", i)

    ref = " ".join(lines[i:j])
    del lines[i : j + 1]

    # play safe, clean empty lines
    while True:
        if lines[i]:
            break
        del lines[i]

    j = find_token(lines, "\\end_inset", i - 1)
    if i == j:
        del lines[i]
    else:
        document.warning("Unexpected end of inset.")
    j = find_token(lines, "\\begin_inset LatexDel }", i)
    label = " ".join(lines[i:j])
    del lines[i : j + 1]

    return ref, label


def update_ref(document):
    "Update reference inset."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\begin_inset LatexCommand", i)
        if i == -1:
            return

        if lines[i].split()[-1] == "\\ref{":
            i = i + 1
            ref, label = latexdel_getargs(document, i)
            lines[i - 1] = f"{lines[i - 1][:-1]}[{ref}]{{{label}}}"

        i = i + 1


def update_latexdel(document):
    "Remove latexdel insets."
    lines = document.body
    i = 0
    latexdel_re = re.compile(r".*\\begin_inset LatexDel")
    while True:
        i = find_re(lines, latexdel_re, i)
        if i == -1:
            return
        lines[i] = lines[i].replace("\\begin_inset LatexDel", "\\begin_inset LatexCommand")

        j = lines[i].find("\\begin_inset")
        lines.insert(i + 1, lines[i][j:])
        lines[i] = lines[i][:j].strip()
        i = i + 1

        if lines[i].split()[-1] in ("\\url{", "\\htmlurl{"):
            i = i + 1

            ref, label = latexdel_getargs(document, i)
            lines[i - 1] = f"{lines[i-1][:-1]}[{label}]{{{ref}}}"

        i = i + 1


supported_versions = ["1.1.5", "1.1.5fix1", "1.1.5fix2", "1.1"]
convert = [
    [
        216,
        [
            first_layout,
            remove_vcid,
            remove_cursor,
            update_toc,
            replace_protected_separator,
            merge_formula_inset,
            update_tabular,
            remove_space_in_units,
            update_ref,
            update_latexdel,
        ],
    ]
]

revert = []

if __name__ == "__main__":
    pass
