/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/
#include "curl_setup.h"

#if !defined(CURL_DISABLE_HTTP) && !defined(CURL_DISABLE_COOKIES)

#include "urldata.h"
#include "cookie.h"
#include "psl.h"
#include "curl_trc.h"
#include "slist.h"
#include "curl_share.h"
#include "strcase.h"
#include "curl_fopen.h"
#include "curl_get_line.h"
#include "curl_memrchr.h"
#include "parsedate.h"
#include "strdup.h"
#include "llist.h"
#include "bufref.h"
#include "curlx/strparse.h"

/* number of seconds in 400 days */
#define COOKIES_MAXAGE (400 * 24 * 3600)

/* Make sure cookies never expire further away in time than 400 days into the
   future. (from RFC6265bis draft-19)

   For the sake of easier testing, align the capped time to an even 60 second
   boundary.
*/
static void cap_expires(time_t now, struct Cookie *co)
{
  if(co->expires && (TIME_T_MAX - COOKIES_MAXAGE - 30) > now) {
    timediff_t cap = now + COOKIES_MAXAGE;
    if(co->expires > cap) {
      cap += 30;
      co->expires = (cap / 60) * 60;
    }
  }
}

static void freecookie(struct Cookie *co, bool maintoo)
{
  curlx_free(co->domain);
  curlx_free(co->path);
  curlx_free(co->name);
  curlx_free(co->value);
  if(maintoo)
    curlx_free(co);
}

static bool cookie_tailmatch(const char *cookie_domain,
                             size_t cookie_domain_len,
                             const char *hostname)
{
  size_t hostname_len = strlen(hostname);

  if(hostname_len < cookie_domain_len)
    return FALSE;

  if(!curl_strnequal(cookie_domain,
                     hostname + hostname_len - cookie_domain_len,
                     cookie_domain_len))
    return FALSE;

  /*
   * A lead char of cookie_domain is not '.'.
   * RFC6265 4.1.2.3. The Domain Attribute says:
   * For example, if the value of the Domain attribute is
   * "example.com", the user agent will include the cookie in the Cookie
   * header when making HTTP requests to example.com, www.example.com, and
   * www.corp.example.com.
   */
  if(hostname_len == cookie_domain_len)
    return TRUE;
  if('.' == *(hostname + hostname_len - cookie_domain_len - 1))
    return TRUE;
  return FALSE;
}

/*
 * matching cookie path and URL path
 * RFC6265 5.1.4 Paths and Path-Match
 */
static bool pathmatch(const char *cookie_path, const char *uri_path)
{
  size_t cookie_path_len;
  size_t uri_path_len;
  bool ret = FALSE;

  /* cookie_path must not have last '/' separator. ex: /sample */
  cookie_path_len = strlen(cookie_path);
  if(cookie_path_len == 1) {
    /* cookie_path must be '/' */
    return TRUE;
  }

  /* #-fragments are already cut off! */
  if(strlen(uri_path) == 0 || uri_path[0] != '/')
    uri_path = "/";

  /*
   * here, RFC6265 5.1.4 says
   *  4. Output the characters of the uri-path from the first character up
   *     to, but not including, the right-most %x2F ("/").
   *  but URL path /hoge?fuga=xxx means /hoge/index.cgi?fuga=xxx in some site
   *  without redirect.
   *  Ignore this algorithm because /hoge is uri path for this case
   *  (uri path is not /).
   */

  uri_path_len = strlen(uri_path);

  if(uri_path_len < cookie_path_len)
    goto pathmatched;

  /* not using checkprefix() because matching should be case-sensitive */
  if(strncmp(cookie_path, uri_path, cookie_path_len))
    goto pathmatched;

  /* The cookie-path and the uri-path are identical. */
  if(cookie_path_len == uri_path_len) {
    ret = TRUE;
    goto pathmatched;
  }

  /* here, cookie_path_len < uri_path_len */
  if(uri_path[cookie_path_len] == '/') {
    ret = TRUE;
    goto pathmatched;
  }

pathmatched:
  return ret;
}

/*
 * Return the top-level domain, for optimal hashing.
 */
static const char *get_top_domain(const char * const domain, size_t *outlen)
{
  size_t len = 0;
  const char *first = NULL, *last;

  if(domain) {
    len = strlen(domain);
    last = memrchr(domain, '.', len);
    if(last) {
      first = memrchr(domain, '.', (last - domain));
      if(first)
        len -= (++first - domain);
    }
  }

  if(outlen)
    *outlen = len;

  return first ? first : domain;
}

/* Avoid C1001, an "internal error" with MSVC14 */
#if defined(_MSC_VER) && (_MSC_VER == 1900)
#pragma optimize("", off)
#endif

/*
 * A case-insensitive hash for the cookie domains.
 */
static size_t cookie_hash_domain(const char *domain, const size_t len)
{
  const char *end = domain + len;
  size_t h = 5381;

  while(domain < end) {
    size_t j = (size_t)Curl_raw_toupper(*domain++);
    h += h << 5;
    h ^= j;
  }

  return (h % COOKIE_HASH_SIZE);
}

#if defined(_MSC_VER) && (_MSC_VER == 1900)
#pragma optimize("", on)
#endif

/*
 * Hash this domain.
 */
static size_t cookiehash(const char * const domain)
{
  const char *top;
  size_t len;

  if(!domain || Curl_host_is_ipnum(domain))
    return 0;

  top = get_top_domain(domain, &len);
  return cookie_hash_domain(top, len);
}

/*
 * cookie path sanitize
 */
static char *sanitize_cookie_path(const char *cookie_path, size_t len)
{
  /* some sites send path attribute within '"'. */
  if(len && (cookie_path[0] == '\"')) {
    cookie_path++;
    len--;

    if(len && (cookie_path[len - 1] == '\"'))
      len--;
  }

  /* RFC6265 5.2.4 The Path Attribute */
  if(!len || (cookie_path[0] != '/'))
    /* Let cookie-path be the default-path. */
    return curlx_strdup("/");

  /* remove trailing slash when path is non-empty */
  /* convert /hoge/ to /hoge */
  if(len > 1 && cookie_path[len - 1] == '/')
    len--;

  return Curl_memdup0(cookie_path, len);
}

/*
 * strstore
 *
 * A thin wrapper around strdup which ensures that any memory allocated at
 * *str will be freed before the string allocated by strdup is stored there.
 * The intended usecase is repeated assignments to the same variable during
 * parsing in a last-wins scenario. The caller is responsible for checking
 * for OOM errors.
 */
static CURLcode strstore(char **str, const char *newstr, size_t len)
{
  DEBUGASSERT(str);
  if(!len) {
    len++;
    newstr = "";
  }
  *str = Curl_memdup0(newstr, len);
  if(!*str)
    return CURLE_OUT_OF_MEMORY;
  return CURLE_OK;
}

/*
 * remove_expired
 *
 * Remove expired cookies from the hash by inspecting the expires timestamp on
 * each cookie in the hash, freeing and deleting any where the timestamp is in
 * the past. If the cookiejar has recorded the next timestamp at which one or
 * more cookies expire, then processing will exit early in case this timestamp
 * is in the future.
 */
static void remove_expired(struct CookieInfo *ci)
{
  struct Cookie *co;
  curl_off_t now = (curl_off_t)time(NULL);
  unsigned int i;

  /*
   * If the earliest expiration timestamp in the jar is in the future we can
   * skip scanning the whole jar and instead exit early as there will not be
   * any cookies to evict. If we need to evict however, reset the
   * next_expiration counter in order to track the next one. In case the
   * recorded first expiration is the max offset, then perform the safe
   * fallback of checking all cookies.
   */
  if(now < ci->next_expiration &&
     ci->next_expiration != CURL_OFF_T_MAX)
    return;
  else
    ci->next_expiration = CURL_OFF_T_MAX;

  for(i = 0; i < COOKIE_HASH_SIZE; i++) {
    struct Curl_llist_node *n;
    struct Curl_llist_node *e = NULL;

    for(n = Curl_llist_head(&ci->cookielist[i]); n; n = e) {
      co = Curl_node_elem(n);
      e = Curl_node_next(n);
      if(co->expires) {
        if(co->expires < now) {
          Curl_node_remove(n);
          freecookie(co, TRUE);
          ci->numcookies--;
        }
        else if(co->expires < ci->next_expiration)
          /*
           * If this cookie has an expiration timestamp earlier than what we
           * have seen so far then record it for the next round of expirations.
           */
          ci->next_expiration = co->expires;
      }
    }
  }
}

#ifndef USE_LIBPSL
/* Make sure domain contains a dot or is localhost. */
static bool bad_domain(const char *domain, size_t len)
{
  if((len == 9) && curl_strnequal(domain, "localhost", 9))
    return FALSE;
  else {
    /* there must be a dot present, but that dot must not be a trailing dot */
    char *dot = memchr(domain, '.', len);
    if(dot) {
      size_t i = dot - domain;
      if((len - i) > 1)
        /* the dot is not the last byte */
        return FALSE;
    }
  }
  return TRUE;
}
#endif

/*
  RFC 6265 section 4.1.1 says a server should accept this range:

  cookie-octet    = %x21 / %x23-2B / %x2D-3A / %x3C-5B / %x5D-7E

  But Firefox and Chrome as of June 2022 accept space, comma and double-quotes
  fine. The prime reason for filtering out control bytes is that some HTTP
  servers return 400 for requests that contain such.
*/
static bool invalid_octets(const char *ptr, size_t len)
{
  const unsigned char *p = (const unsigned char *)ptr;
  /* Reject all bytes \x01 - \x1f (*except* \x09, TAB) + \x7f */
  while(len && *p) {
    if(((*p != 9) && (*p < 0x20)) || (*p == 0x7f))
      return TRUE;
    p++;
    len--;
  }
  return FALSE;
}

/* The maximum length we accept a date string for the 'expire' keyword. The
   standard date formats are within the 30 bytes range. This adds an extra
   margin just to make sure it realistically works with what is used out
   there.
*/
#define MAX_DATE_LENGTH 80

#define COOKIE_NAME 0
#define COOKIE_VALUE 1
#define COOKIE_DOMAIN 2
#define COOKIE_PATH 3

#define COOKIE_PIECES 4 /* the list above */

static CURLcode storecookie(struct Cookie *co, struct Curl_str *cp,
                            const char *path, const char *domain)
{
  CURLcode result;
  result = strstore(&co->name, curlx_str(&cp[COOKIE_NAME]),
                    curlx_strlen(&cp[COOKIE_NAME]));
  if(!result)
    result = strstore(&co->value, curlx_str(&cp[COOKIE_VALUE]),
                      curlx_strlen(&cp[COOKIE_VALUE]));
  if(!result) {
    size_t plen = 0;
    if(curlx_strlen(&cp[COOKIE_PATH])) {
      path = curlx_str(&cp[COOKIE_PATH]);
      plen = curlx_strlen(&cp[COOKIE_PATH]);
    }
    else if(path) {
      /* No path was given in the header line, set the default */
      const char *endslash = strrchr(path, '/');
      if(endslash)
        plen = (endslash - path + 1); /* include end slash */
      else
        plen = strlen(path);
    }

    if(path) {
      co->path = sanitize_cookie_path(path, plen);
      if(!co->path)
        result = CURLE_OUT_OF_MEMORY;
    }
  }
  if(!result) {
    if(curlx_strlen(&cp[COOKIE_DOMAIN]))
      result = strstore(&co->domain, curlx_str(&cp[COOKIE_DOMAIN]),
                        curlx_strlen(&cp[COOKIE_DOMAIN]));
    else if(domain) {
      /* no domain was given in the header line, set the default */
      co->domain = curlx_strdup(domain);
      if(!co->domain)
        result = CURLE_OUT_OF_MEMORY;
    }
  }
  return result;
}

/* this function return errors on OOM etc, not on plain cookie format
   problems */
static CURLcode
parse_cookie_header(struct Curl_easy *data,
                    struct Cookie *co,
                    struct CookieInfo *ci,
                    bool *okay, /* if the cookie was fine */
                    const char *ptr,
                    const char *domain, /* default domain */
                    const char *path,   /* full path used when this cookie is
                                           set, used to get default path for
                                           the cookie unless set */
                    bool secure)  /* TRUE if connection is over secure
                                     origin */
{
  /* This line was read off an HTTP-header */
  time_t now = 0;
  size_t linelength = strlen(ptr);
  CURLcode result = CURLE_OK;
  struct Curl_str cookie[COOKIE_PIECES];
  *okay = FALSE;
  if(linelength > MAX_COOKIE_LINE)
    /* discard overly long lines at once */
    return CURLE_OK;

  /* memset instead of initializer because gcc 4.8.1 is silly */
  memset(cookie, 0, sizeof(cookie));
  do {
    struct Curl_str name;
    struct Curl_str val;

    /* we have a <name>=<value> pair or a stand-alone word here */
    if(!curlx_str_cspn(&ptr, &name, ";\t\r\n=")) {
      bool sep = FALSE;
      curlx_str_trimblanks(&name);

      if(!curlx_str_single(&ptr, '=')) {
        sep = TRUE; /* a '=' was used */
        if(!curlx_str_cspn(&ptr, &val, ";\r\n"))
          curlx_str_trimblanks(&val);
      }
      else
        curlx_str_init(&val);

      if(!curlx_strlen(&cookie[COOKIE_NAME])) {
        /* The first name/value pair is the actual cookie name */
        if(!sep ||
           /* Bad name/value pair. */
           invalid_octets(curlx_str(&name), curlx_strlen(&name)) ||
           invalid_octets(curlx_str(&val), curlx_strlen(&val)) ||
           !curlx_strlen(&name)) {
          infof(data, "invalid octets in name/value, cookie dropped");
          return CURLE_OK;
        }

        /*
         * Check for too long individual name or contents, or too long
         * combination of name + contents. Chrome and Firefox support 4095 or
         * 4096 bytes combo
         */
        if(curlx_strlen(&name) >= (MAX_NAME - 1) ||
           curlx_strlen(&val) >= (MAX_NAME - 1) ||
           ((curlx_strlen(&name) + curlx_strlen(&val)) > MAX_NAME)) {
          infof(data, "oversized cookie dropped, name/val %zu + %zu bytes",
                curlx_strlen(&name), curlx_strlen(&val));
          return CURLE_OK;
        }

        /* Reject cookies with a TAB inside the value */
        if(curlx_strlen(&val) &&
           memchr(curlx_str(&val), '\t', curlx_strlen(&val))) {
          infof(data, "cookie contains TAB, dropping");
          return CURLE_OK;
        }

        /* Check if we have a reserved prefix set. */
        if(!strncmp("__Secure-", curlx_str(&name), 9))
          co->prefix_secure = TRUE;
        else if(!strncmp("__Host-", curlx_str(&name), 7))
          co->prefix_host = TRUE;

        cookie[COOKIE_NAME] = name;
        cookie[COOKIE_VALUE] = val;
      }
      else if(!sep) {
        /*
         * this is a "<name>" with no content
         */

        /*
         * secure cookies are only allowed to be set when the connection is
         * using a secure protocol, or when the cookie is being set by
         * reading from file
         */
        if(curlx_str_casecompare(&name, "secure")) {
          if(secure || !ci->running)
            co->secure = TRUE;
          else {
            infof(data, "skipped cookie because not 'secure'");
            return CURLE_OK;
          }
        }
        else if(curlx_str_casecompare(&name, "httponly"))
          co->httponly = TRUE;
      }
      else if(curlx_str_casecompare(&name, "path")) {
        cookie[COOKIE_PATH] = val;
      }
      else if(curlx_str_casecompare(&name, "domain") && curlx_strlen(&val)) {
        bool is_ip;
        const char *v = curlx_str(&val);
        /*
         * Now, we make sure that our host is within the given domain, or
         * the given domain is not valid and thus cannot be set.
         */

        if('.' == *v)
          curlx_str_nudge(&val, 1);

#ifndef USE_LIBPSL
        /*
         * Without PSL we do not know when the incoming cookie is set on a
         * TLD or otherwise "protected" suffix. To reduce risk, we require a
         * dot OR the exact hostname being "localhost".
         */
        if(bad_domain(curlx_str(&val), curlx_strlen(&val)))
          domain = ":";
#endif

        is_ip = Curl_host_is_ipnum(domain ? domain : curlx_str(&val));

        if(!domain ||
           (is_ip &&
            !strncmp(curlx_str(&val), domain, curlx_strlen(&val)) &&
            (curlx_strlen(&val) == strlen(domain))) ||
           (!is_ip && cookie_tailmatch(curlx_str(&val),
                                          curlx_strlen(&val), domain))) {
          cookie[COOKIE_DOMAIN] = val;
          if(!is_ip)
            co->tailmatch = TRUE; /* we always do that if the domain name was
                                     given */
        }
        else {
          /*
           * We did not get a tailmatch and then the attempted set domain is
           * not a domain to which the current host belongs. Mark as bad.
           */
          infof(data, "skipped cookie with bad tailmatch domain: %s",
                curlx_str(&val));
          return CURLE_OK;
        }
      }
      else if(curlx_str_casecompare(&name, "max-age") && curlx_strlen(&val)) {
        /*
         * Defined in RFC2109:
         *
         * Optional. The Max-Age attribute defines the lifetime of the
         * cookie, in seconds. The delta-seconds value is a decimal non-
         * negative integer. After delta-seconds seconds elapse, the
         * client should discard the cookie. A value of zero means the
         * cookie should be discarded immediately.
         */
        int rc;
        const char *maxage = curlx_str(&val);
        if(*maxage == '\"')
          maxage++;
        rc = curlx_str_number(&maxage, &co->expires, CURL_OFF_T_MAX);
        if(!now)
          now = time(NULL);
        switch(rc) {
        case STRE_OVERFLOW:
          /* overflow, used max value */
          co->expires = CURL_OFF_T_MAX;
          break;
        default:
          /* negative or otherwise bad, expire */
          co->expires = 1;
          break;
        case STRE_OK:
          if(!co->expires)
            co->expires = 1; /* expire now */
          else if(CURL_OFF_T_MAX - now < co->expires)
            /* would overflow */
            co->expires = CURL_OFF_T_MAX;
          else
            co->expires += now;
          break;
        }
        cap_expires(now, co);
      }
      else if(curlx_str_casecompare(&name, "expires") && curlx_strlen(&val) &&
              !co->expires && (curlx_strlen(&val) < MAX_DATE_LENGTH)) {
        /*
         * Let max-age have priority.
         *
         * If the date cannot get parsed for whatever reason, the cookie
         * will be treated as a session cookie
         */
        char dbuf[MAX_DATE_LENGTH + 1];
        time_t date = 0;
        memcpy(dbuf, curlx_str(&val), curlx_strlen(&val));
        dbuf[curlx_strlen(&val)] = 0;
        if(!Curl_getdate_capped(dbuf, &date)) {
          if(!date)
            date++;
          co->expires = (curl_off_t)date;
        }
        else
          co->expires = 0;
        if(!now)
          now = time(NULL);
        cap_expires(now, co);
      }
    }
  } while(!curlx_str_single(&ptr, ';'));

  if(curlx_strlen(&cookie[COOKIE_NAME])) {
    /* the header was fine, now store the data */
    result = storecookie(co, &cookie[0], path, domain);
    if(!result)
      *okay = TRUE;
  }
  return result;
}

static CURLcode parse_netscape(struct Cookie *co,
                               struct CookieInfo *ci,
                               bool *okay,
                               const char *lineptr,
                               bool secure) /* TRUE if connection is over
                                               secure origin */
{
  /*
   * This line is NOT an HTTP header style line, we do offer support for
   * reading the odd netscape cookies-file format here
   */
  const char *ptr, *next;
  int fields;
  size_t len;
  *okay = FALSE;

  /*
   * In 2008, Internet Explorer introduced HTTP-only cookies to prevent XSS
   * attacks. Cookies marked httpOnly are not accessible to JavaScript. In
   * Firefox's cookie files, they are prefixed #HttpOnly_ and the rest
   * remains as usual, so we skip 10 characters of the line.
   */
  if(strncmp(lineptr, "#HttpOnly_", 10) == 0) {
    lineptr += 10;
    co->httponly = TRUE;
  }

  if(lineptr[0] == '#')
    /* do not even try the comments */
    return CURLE_OK;

  /*
   * Now loop through the fields and init the struct we already have
   * allocated
   */
  fields = 0;
  for(next = lineptr; next; fields++) {
    ptr = next;
    len = strcspn(ptr, "\t\r\n");
    next = (ptr[len] == '\t' ? &ptr[len + 1] : NULL);
    switch(fields) {
    case 0:
      if(ptr[0] == '.') { /* skip preceding dots */
        ptr++;
        len--;
      }
      co->domain = Curl_memdup0(ptr, len);
      if(!co->domain)
        return CURLE_OUT_OF_MEMORY;
      break;
    case 1:
      /*
       * flag: A TRUE/FALSE value indicating if all machines within a given
       * domain can access the variable. Set TRUE when the cookie says
       * .example.com and to false when the domain is complete www.example.com
       */
      co->tailmatch = !!curl_strnequal(ptr, "TRUE", len);
      break;
    case 2:
      /* The file format allows the path field to remain not filled in */
      if(strncmp("TRUE", ptr, len) && strncmp("FALSE", ptr, len)) {
        /* only if the path does not look like a boolean option! */
        co->path = sanitize_cookie_path(ptr, len);
        if(!co->path)
          return CURLE_OUT_OF_MEMORY;
        break;
      }
      else {
        /* this does not look like a path, make one up! */
        co->path = curlx_strdup("/");
        if(!co->path)
          return CURLE_OUT_OF_MEMORY;
      }
      fields++; /* add a field and fall down to secure */
      FALLTHROUGH();
    case 3:
      co->secure = FALSE;
      if(curl_strnequal(ptr, "TRUE", len)) {
        if(secure || ci->running)
          co->secure = TRUE;
        else
          return CURLE_OK;
      }
      break;
    case 4:
      if(curlx_str_number(&ptr, &co->expires, CURL_OFF_T_MAX))
        return CURLE_OK;
      break;
    case 5:
      co->name = Curl_memdup0(ptr, len);
      if(!co->name)
        return CURLE_OUT_OF_MEMORY;
      else {
        /* For Netscape file format cookies we check prefix on the name */
        if(curl_strnequal("__Secure-", co->name, 9))
          co->prefix_secure = TRUE;
        else if(curl_strnequal("__Host-", co->name, 7))
          co->prefix_host = TRUE;
      }
      break;
    case 6:
      co->value = Curl_memdup0(ptr, len);
      if(!co->value)
        return CURLE_OUT_OF_MEMORY;
      break;
    }
  }
  if(fields == 6) {
    /* we got a cookie with blank contents, fix it */
    co->value = curlx_strdup("");
    if(!co->value)
      return CURLE_OUT_OF_MEMORY;
    else
      fields++;
  }

  if(fields != 7)
    /* we did not find the sufficient number of fields */
    return CURLE_OK;

  *okay = TRUE;
  return CURLE_OK;
}

static bool is_public_suffix(struct Curl_easy *data,
                             struct Cookie *co,
                             const char *domain)
{
#ifdef USE_LIBPSL
  /*
   * Check if the domain is a Public Suffix and if yes, ignore the cookie. We
   * must also check that the data handle is not NULL since the psl code will
   * dereference it.
   */
  DEBUGF(infof(data, "PSL check set-cookie '%s' for domain=%s in %s",
         co->name, co->domain, domain));
  if(data && (domain && co->domain && !Curl_host_is_ipnum(co->domain))) {
    bool acceptable = FALSE;
    char lcase[256];
    char lcookie[256];
    size_t dlen = strlen(domain);
    size_t clen = strlen(co->domain);
    if((dlen < sizeof(lcase)) && (clen < sizeof(lcookie))) {
      const psl_ctx_t *psl = Curl_psl_use(data);
      if(psl) {
        /* the PSL check requires lowercase domain name and pattern */
        Curl_strntolower(lcase, domain, dlen + 1);
        Curl_strntolower(lcookie, co->domain, clen + 1);
        acceptable = psl_is_cookie_domain_acceptable(psl, lcase, lcookie);
        Curl_psl_release(data);
      }
      else
        infof(data, "libpsl problem, rejecting cookie for safety");
    }

    if(!acceptable) {
      infof(data, "cookie '%s' dropped, domain '%s' must not "
            "set cookies for '%s'", co->name, domain, co->domain);
      return TRUE;
    }
  }
#else
  (void)data;
  (void)co;
  (void)domain;
  DEBUGF(infof(data, "NO PSL to check set-cookie '%s' for domain=%s in %s",
         co->name, co->domain, domain));
#endif
  return FALSE;
}

/* returns TRUE when replaced */
static bool replace_existing(struct Curl_easy *data,
                             struct Cookie *co,
                             struct CookieInfo *ci,
                             bool secure,
                             bool *replacep)
{
  bool replace_old = FALSE;
  struct Curl_llist_node *replace_n = NULL;
  struct Curl_llist_node *n;
  size_t myhash = cookiehash(co->domain);
  for(n = Curl_llist_head(&ci->cookielist[myhash]); n; n = Curl_node_next(n)) {
    struct Cookie *clist = Curl_node_elem(n);
    if(!strcmp(clist->name, co->name)) {
      /* the names are identical */
      bool matching_domains = FALSE;

      if(clist->domain && co->domain) {
        if(curl_strequal(clist->domain, co->domain))
          /* The domains are identical */
          matching_domains = TRUE;
      }
      else if(!clist->domain && !co->domain)
        matching_domains = TRUE;

      if(matching_domains && /* the domains were identical */
         clist->path && co->path && /* both have paths */
         clist->secure && !co->secure && !secure) {
        size_t cllen;
        const char *sep = NULL;

        /*
         * A non-secure cookie may not overlay an existing secure cookie.
         * For an existing cookie "a" with path "/login", refuse a new
         * cookie "a" with for example path "/login/en", while the path
         * "/loginhelper" is ok.
         */

        DEBUGASSERT(clist->path[0]);
        if(clist->path[0])
          sep = strchr(clist->path + 1, '/');
        if(sep)
          cllen = sep - clist->path;
        else
          cllen = strlen(clist->path);

        if(curl_strnequal(clist->path, co->path, cllen)) {
          infof(data, "cookie '%s' for domain '%s' dropped, would "
                "overlay an existing cookie", co->name, co->domain);
          return FALSE;
        }
      }
    }

    if(!replace_n && !strcmp(clist->name, co->name)) {
      /* the names are identical */

      if(clist->domain && co->domain) {
        if(curl_strequal(clist->domain, co->domain) &&
           (clist->tailmatch == co->tailmatch))
          /* The domains are identical */
          replace_old = TRUE;
      }
      else if(!clist->domain && !co->domain)
        replace_old = TRUE;

      if(replace_old) {
        /* the domains were identical */

        if(clist->path && co->path &&
           !curl_strequal(clist->path, co->path))
          replace_old = FALSE;
        else if(!clist->path != !co->path)
          replace_old = FALSE;
      }

      if(replace_old && !co->livecookie && clist->livecookie) {
        /*
         * Both cookies matched fine, except that the already present cookie
         * is "live", which means it was set from a header, while the new one
         * was read from a file and thus is not "live". "live" cookies are
         * preferred so the new cookie is freed.
         */
        return FALSE;
      }
      if(replace_old)
        replace_n = n;
    }
  }
  if(replace_n) {
    struct Cookie *repl = Curl_node_elem(replace_n);

    /* when replacing, creationtime is kept from old */
    co->creationtime = repl->creationtime;

    /* unlink the old */
    Curl_node_remove(replace_n);

    /* free the old cookie */
    freecookie(repl, TRUE);
  }
  *replacep = replace_old;
  return TRUE;
}

/*
 * Curl_cookie_add
 *
 * Add a single cookie line to the cookie keeping object. Be aware that
 * sometimes we get an IP-only hostname, and that might also be a numerical
 * IPv6 address.
 *
 */
CURLcode
Curl_cookie_add(struct Curl_easy *data,
                struct CookieInfo *ci,
                bool httpheader, /* TRUE if HTTP header-style line */
                bool noexpire, /* if TRUE, skip remove_expired() */
                const char *lineptr,   /* first character of the line */
                const char *domain, /* default domain */
                const char *path,   /* full path used when this cookie is set,
                                       used to get default path for the cookie
                                       unless set */
                bool secure)  /* TRUE if connection is over secure origin */
{
  struct Cookie comem;
  struct Cookie *co;
  size_t myhash;
  CURLcode result;
  bool replaces = FALSE;
  bool okay;

  DEBUGASSERT(data);
  DEBUGASSERT(MAX_SET_COOKIE_AMOUNT <= 255); /* counter is an unsigned char */
  if(data->req.setcookies >= MAX_SET_COOKIE_AMOUNT)
    return CURLE_OK; /* silently ignore */

  co = &comem;
  memset(co, 0, sizeof(comem));

  if(httpheader)
    result = parse_cookie_header(data, co, ci, &okay,
                                 lineptr, domain, path, secure);
  else
    result = parse_netscape(co, ci, &okay, lineptr, secure);

  if(result || !okay)
    goto fail;

  if(co->prefix_secure && !co->secure)
    /* The __Secure- prefix only requires that the cookie be set secure */
    goto fail;

  if(co->prefix_host) {
    /*
     * The __Host- prefix requires the cookie to be secure, have a "/" path
     * and not have a domain set.
     */
    if(co->secure && co->path && !strcmp(co->path, "/") && !co->tailmatch)
      ;
    else
      goto fail;
  }

  if(!ci->running &&    /* read from a file */
     ci->newsession &&  /* clean session cookies */
     !co->expires)      /* this is a session cookie */
    goto fail;

  co->livecookie = ci->running;
  co->creationtime = ++ci->lastct;

  /*
   * Now we have parsed the incoming line, we must now check if this supersedes
   * an already existing cookie, which it may if the previous have the same
   * domain and path as this.
   */

  /* remove expired cookies */
  if(!noexpire)
    remove_expired(ci);

  if(is_public_suffix(data, co, domain))
    goto fail;

  if(!replace_existing(data, co, ci, secure, &replaces))
    goto fail;

  /* clone the stack struct into heap */
  co = Curl_memdup(&comem, sizeof(comem));
  if(!co) {
    co = &comem;
    result = CURLE_OUT_OF_MEMORY;
    goto fail; /* bail out if we are this low on memory */
  }

  /* add this cookie to the list */
  myhash = cookiehash(co->domain);
  Curl_llist_append(&ci->cookielist[myhash], co, &co->node);

  if(ci->running)
    /* Only show this when NOT reading the cookies from a file */
    infof(data, "%s cookie %s=\"%s\" for domain %s, path %s, "
          "expire %" FMT_OFF_T,
          replaces ? "Replaced" : "Added", co->name, co->value,
          co->domain, co->path, co->expires);

  if(!replaces)
    ci->numcookies++; /* one more cookie in the jar */

  /*
   * Now that we have added a new cookie to the jar, update the expiration
   * tracker in case it is the next one to expire.
   */
  if(co->expires && (co->expires < ci->next_expiration))
    ci->next_expiration = co->expires;

  if(httpheader)
    data->req.setcookies++;

  return result;
fail:
  freecookie(co, FALSE);
  return result;
}

/*
 * Curl_cookie_init()
 *
 * Inits a cookie struct to read data from a local file. This is always
 * called before any cookies are set. File may be NULL in which case only the
 * struct is initialized. Is file is "-" then STDIN is read.
 *
 * If 'newsession' is TRUE, discard all "session cookies" on read from file.
 *
 * Note that 'data' might be called as NULL pointer. If data is NULL, 'file'
 * will be ignored.
 *
 * Returns NULL on out of memory.
 */
struct CookieInfo *Curl_cookie_init(void)
{
  int i;
  struct CookieInfo *ci = curlx_calloc(1, sizeof(struct CookieInfo));
  if(!ci)
    return NULL;

  /* This does not use the destructor callback since we want to add
     and remove to lists while keeping the cookie struct intact */
  for(i = 0; i < COOKIE_HASH_SIZE; i++)
    Curl_llist_init(&ci->cookielist[i], NULL);
  /*
   * Initialize the next_expiration time to signal that we do not have enough
   * information yet.
   */
  ci->next_expiration = CURL_OFF_T_MAX;

  return ci;
}

/*
 * cookie_load()
 *
 * Reads cookies from a local file. This is always called before any cookies
 * are set. If file is "-" then STDIN is read.
 *
 * If 'newsession' is TRUE, discard all "session cookies" on read from file.
 *
 */
static CURLcode cookie_load(struct Curl_easy *data, const char *file,
                            struct CookieInfo *ci, bool newsession)
{
  FILE *handle = NULL;
  CURLcode result = CURLE_OK;
  FILE *fp = NULL;
  DEBUGASSERT(ci);
  DEBUGASSERT(data);
  DEBUGASSERT(file);

  ci->newsession = newsession; /* new session? */
  ci->running = FALSE; /* this is not running, this is init */

  if(file && *file) {
    if(!strcmp(file, "-"))
      fp = stdin;
    else {
      fp = curlx_fopen(file, "rb");
      if(!fp)
        infof(data, "WARNING: failed to open cookie file \"%s\"", file);
      else
        handle = fp;
    }
  }

  if(fp) {
    struct dynbuf buf;
    bool eof = FALSE;
    curlx_dyn_init(&buf, MAX_COOKIE_LINE);
    do {
      result = Curl_get_line(&buf, fp, &eof);
      if(!result) {
        const char *lineptr = curlx_dyn_ptr(&buf);
        bool headerline = FALSE;
        if(checkprefix("Set-Cookie:", lineptr)) {
          /* This is a cookie line, get it! */
          lineptr += 11;
          headerline = TRUE;
          curlx_str_passblanks(&lineptr);
        }

        result = Curl_cookie_add(data, ci, headerline, TRUE, lineptr, NULL,
                                 NULL, TRUE);
        /* File reading cookie failures are not propagated back to the
           caller because there is no way to do that */
      }
    } while(!result && !eof);
    curlx_dyn_free(&buf); /* free the line buffer */

    /*
     * Remove expired cookies from the hash. We must make sure to run this
     * after reading the file, and not on every cookie.
     */
    remove_expired(ci);

    if(handle)
      curlx_fclose(handle);
  }
  data->state.cookie_engine = TRUE;
  ci->running = TRUE;          /* now, we are running */

  return result;
}

/*
 * Load cookies from all given cookie files (CURLOPT_COOKIEFILE).
 */
CURLcode Curl_cookie_loadfiles(struct Curl_easy *data)
{
  CURLcode result = CURLE_OK;
  struct curl_slist *list = data->state.cookielist;
  if(list) {
    Curl_share_lock(data, CURL_LOCK_DATA_COOKIE, CURL_LOCK_ACCESS_SINGLE);
    if(!data->cookies)
      data->cookies = Curl_cookie_init();
    if(!data->cookies)
      result = CURLE_OUT_OF_MEMORY;
    else {
      data->state.cookie_engine = TRUE;
      while(list) {
        result = cookie_load(data, list->data, data->cookies,
                             data->set.cookiesession);
        if(result)
          break;
        list = list->next;
      }
    }
    Curl_share_unlock(data, CURL_LOCK_DATA_COOKIE);
  }
  return result;
}

/*
 * cookie_sort
 *
 * Helper function to sort cookies such that the longest path gets before the
 * shorter path. Path, domain and name lengths are considered in that order,
 * with the creationtime as the tiebreaker. The creationtime is guaranteed to
 * be unique per cookie, so we know we will get an ordering at that point.
 */
static int cookie_sort(const void *p1, const void *p2)
{
  const struct Cookie *c1 = *(const struct Cookie * const *)p1;
  const struct Cookie *c2 = *(const struct Cookie * const *)p2;
  size_t l1, l2;

  /* 1 - compare cookie path lengths */
  l1 = c1->path ? strlen(c1->path) : 0;
  l2 = c2->path ? strlen(c2->path) : 0;

  if(l1 != l2)
    return (l2 > l1) ? 1 : -1; /* avoid size_t <=> int conversions */

  /* 2 - compare cookie domain lengths */
  l1 = c1->domain ? strlen(c1->domain) : 0;
  l2 = c2->domain ? strlen(c2->domain) : 0;

  if(l1 != l2)
    return (l2 > l1) ? 1 : -1; /* avoid size_t <=> int conversions */

  /* 3 - compare cookie name lengths */
  l1 = c1->name ? strlen(c1->name) : 0;
  l2 = c2->name ? strlen(c2->name) : 0;

  if(l1 != l2)
    return (l2 > l1) ? 1 : -1;

  /* 4 - compare cookie creation time */
  return (c2->creationtime > c1->creationtime) ? 1 : -1;
}

/*
 * cookie_sort_ct
 *
 * Helper function to sort cookies according to creation time.
 */
static int cookie_sort_ct(const void *p1, const void *p2)
{
  const struct Cookie *c1 = *(const struct Cookie * const *)p1;
  const struct Cookie *c2 = *(const struct Cookie * const *)p2;

  return (c2->creationtime > c1->creationtime) ? 1 : -1;
}

bool Curl_secure_context(struct connectdata *conn, const char *host)
{
  return conn->handler->protocol & (CURLPROTO_HTTPS | CURLPROTO_WSS) ||
    curl_strequal("localhost", host) ||
    !strcmp(host, "127.0.0.1") ||
    !strcmp(host, "::1");
}

/*
 * Curl_cookie_getlist
 *
 * For a given host and path, return a linked list of cookies that the client
 * should send to the server if used now. The secure boolean informs the cookie
 * if a secure connection is achieved or not.
 *
 * It shall only return cookies that have not expired.
 *
 * 'okay' is TRUE when there is a list returned.
 */
CURLcode Curl_cookie_getlist(struct Curl_easy *data,
                             struct connectdata *conn,
                             bool *okay,
                             const char *host,
                             struct Curl_llist *list)
{
  size_t matches = 0;
  const bool is_ip = Curl_host_is_ipnum(host);
  const size_t myhash = cookiehash(host);
  struct Curl_llist_node *n;
  const bool secure = Curl_secure_context(conn, host);
  struct CookieInfo *ci = data->cookies;
  const char *path = data->state.up.path;
  CURLcode result = CURLE_OK;
  *okay = FALSE;

  Curl_llist_init(list, NULL);

  if(!ci || !Curl_llist_count(&ci->cookielist[myhash]))
    return CURLE_OK; /* no cookie struct or no cookies in the struct */

  /* at first, remove expired cookies */
  remove_expired(ci);

  for(n = Curl_llist_head(&ci->cookielist[myhash]); n; n = Curl_node_next(n)) {
    struct Cookie *co = Curl_node_elem(n);

    /* if the cookie requires we are secure we must only continue if we are! */
    if(co->secure ? secure : TRUE) {

      /* now check if the domain is correct */
      if(!co->domain ||
         (co->tailmatch && !is_ip &&
          cookie_tailmatch(co->domain, strlen(co->domain), host)) ||
         ((!co->tailmatch || is_ip) && curl_strequal(host, co->domain))) {
        /*
         * the right part of the host matches the domain stuff in the
         * cookie data
         */

        /*
         * now check the left part of the path with the cookies path
         * requirement
         */
        if(!co->path || pathmatch(co->path, path)) {

          /*
           * This is a match and we add it to the return-linked-list
           */
          Curl_llist_append(list, co, &co->getnode);
          matches++;
          if(matches >= MAX_COOKIE_SEND_AMOUNT) {
            infof(data, "Included max number of cookies (%zu) in request!",
                  matches);
            break;
          }
        }
      }
    }
  }

  if(matches) {
    /*
     * Now we need to make sure that if there is a name appearing more than
     * once, the longest specified path version comes first. To make this
     * the swiftest way, we just sort them all based on path length.
     */
    struct Cookie **array;
    size_t i;

    /* alloc an array and store all cookie pointers */
    array = curlx_malloc(sizeof(struct Cookie *) * matches);
    if(!array) {
      result = CURLE_OUT_OF_MEMORY;
      goto fail;
    }

    n = Curl_llist_head(list);

    for(i = 0; n; n = Curl_node_next(n))
      array[i++] = Curl_node_elem(n);

    /* now sort the cookie pointers in path length order */
    qsort(array, matches, sizeof(struct Cookie *), cookie_sort);

    /* remake the linked list order according to the new order */
    Curl_llist_destroy(list, NULL);

    for(i = 0; i < matches; i++)
      Curl_llist_append(list, array[i], &array[i]->getnode);

    curlx_free(array); /* remove the temporary data again */
  }

  *okay = TRUE;
  return CURLE_OK; /* success */

fail:
  /* failure, clear up the allocated chain and return NULL */
  Curl_llist_destroy(list, NULL);
  return result; /* error */
}

/*
 * Curl_cookie_clearall
 *
 * Clear all existing cookies and reset the counter.
 */
void Curl_cookie_clearall(struct CookieInfo *ci)
{
  if(ci) {
    unsigned int i;
    for(i = 0; i < COOKIE_HASH_SIZE; i++) {
      struct Curl_llist_node *n;
      for(n = Curl_llist_head(&ci->cookielist[i]); n;) {
        struct Cookie *c = Curl_node_elem(n);
        struct Curl_llist_node *e = Curl_node_next(n);
        Curl_node_remove(n);
        freecookie(c, TRUE);
        n = e;
      }
    }
    ci->numcookies = 0;
  }
}

/*
 * Curl_cookie_clearsess
 *
 * Free all session cookies in the cookies list.
 */
void Curl_cookie_clearsess(struct CookieInfo *ci)
{
  unsigned int i;

  if(!ci)
    return;

  for(i = 0; i < COOKIE_HASH_SIZE; i++) {
    struct Curl_llist_node *n = Curl_llist_head(&ci->cookielist[i]);
    struct Curl_llist_node *e = NULL;

    for(; n; n = e) {
      struct Cookie *curr = Curl_node_elem(n);
      e = Curl_node_next(n); /* in case the node is removed, get it early */
      if(!curr->expires) {
        Curl_node_remove(n);
        freecookie(curr, TRUE);
        ci->numcookies--;
      }
    }
  }
}

/*
 * Curl_cookie_cleanup()
 *
 * Free a "cookie object" previous created with Curl_cookie_init().
 */
void Curl_cookie_cleanup(struct CookieInfo *ci)
{
  if(ci) {
    Curl_cookie_clearall(ci);
    curlx_free(ci); /* free the base struct as well */
  }
}

/*
 * get_netscape_format()
 *
 * Formats a string for Netscape output file, w/o a newline at the end.
 * Function returns a char * to a formatted line. The caller is responsible
 * for freeing the returned pointer.
 */
static char *get_netscape_format(const struct Cookie *co)
{
  return curl_maprintf(
    "%s"               /* httponly preamble */
    "%s%s\t"           /* domain */
    "%s\t"             /* tailmatch */
    "%s\t"             /* path */
    "%s\t"             /* secure */
    "%" FMT_OFF_T "\t" /* expires */
    "%s\t"             /* name */
    "%s",              /* value */
    co->httponly ? "#HttpOnly_" : "",
    /*
     * Make sure all domains are prefixed with a dot if they allow
     * tailmatching. This is Mozilla-style.
     */
    (co->tailmatch && co->domain && co->domain[0] != '.') ? "." : "",
    co->domain ? co->domain : "unknown",
    co->tailmatch ? "TRUE" : "FALSE",
    co->path ? co->path : "/",
    co->secure ? "TRUE" : "FALSE",
    co->expires,
    co->name,
    co->value ? co->value : "");
}

/*
 * cookie_output()
 *
 * Writes all internally known cookies to the specified file. Specify
 * "-" as filename to write to stdout.
 *
 * The function returns non-zero on write failure.
 */
static CURLcode cookie_output(struct Curl_easy *data,
                              struct CookieInfo *ci,
                              const char *filename)
{
  FILE *out = NULL;
  bool use_stdout = FALSE;
  char *tempstore = NULL;
  CURLcode error = CURLE_OK;

  if(!ci)
    /* no cookie engine alive */
    return CURLE_OK;

  /* at first, remove expired cookies */
  remove_expired(ci);

  if(!strcmp("-", filename)) {
    /* use stdout */
    out = stdout;
    use_stdout = TRUE;
  }
  else {
    error = Curl_fopen(data, filename, &out, &tempstore);
    if(error)
      goto error;
  }

  fputs("# Netscape HTTP Cookie File\n"
        "# https://curl.se/docs/http-cookies.html\n"
        "# This file was generated by libcurl! Edit at your own risk.\n\n",
        out);

  if(ci->numcookies) {
    unsigned int i;
    size_t nvalid = 0;
    struct Cookie **array;
    struct Curl_llist_node *n;

    array = curlx_calloc(1, sizeof(struct Cookie *) * ci->numcookies);
    if(!array) {
      error = CURLE_OUT_OF_MEMORY;
      goto error;
    }

    /* only sort the cookies with a domain property */
    for(i = 0; i < COOKIE_HASH_SIZE; i++) {
      for(n = Curl_llist_head(&ci->cookielist[i]); n; n = Curl_node_next(n)) {
        struct Cookie *co = Curl_node_elem(n);
        if(!co->domain)
          continue;
        array[nvalid++] = co;
      }
    }

    qsort(array, nvalid, sizeof(struct Cookie *), cookie_sort_ct);

    for(i = 0; i < nvalid; i++) {
      char *format_ptr = get_netscape_format(array[i]);
      if(!format_ptr) {
        curlx_free(array);
        error = CURLE_OUT_OF_MEMORY;
        goto error;
      }
      curl_mfprintf(out, "%s\n", format_ptr);
      curlx_free(format_ptr);
    }

    curlx_free(array);
  }

  if(!use_stdout) {
    curlx_fclose(out);
    out = NULL;
    if(tempstore && curlx_rename(tempstore, filename)) {
      error = CURLE_WRITE_ERROR;
      goto error;
    }
  }

  /*
   * If we reach here we have successfully written a cookie file so there is
   * no need to inspect the error, any error case should have jumped into the
   * error block below.
   */
  curlx_free(tempstore);
  return CURLE_OK;

error:
  if(out && !use_stdout)
    curlx_fclose(out);
  if(tempstore) {
    unlink(tempstore);
    curlx_free(tempstore);
  }
  return error;
}

static struct curl_slist *cookie_list(struct Curl_easy *data)
{
  struct curl_slist *list = NULL;
  struct curl_slist *beg;
  unsigned int i;
  struct Curl_llist_node *n;

  if(!data->cookies || (data->cookies->numcookies == 0))
    return NULL;

  /* at first, remove expired cookies */
  remove_expired(data->cookies);

  for(i = 0; i < COOKIE_HASH_SIZE; i++) {
    for(n = Curl_llist_head(&data->cookies->cookielist[i]); n;
        n = Curl_node_next(n)) {
      struct Cookie *c = Curl_node_elem(n);
      char *line;
      if(!c->domain)
        continue;
      line = get_netscape_format(c);
      if(!line) {
        curl_slist_free_all(list);
        return NULL;
      }
      beg = Curl_slist_append_nodup(list, line);
      if(!beg) {
        curlx_free(line);
        curl_slist_free_all(list);
        return NULL;
      }
      list = beg;
    }
  }

  return list;
}

struct curl_slist *Curl_cookie_list(struct Curl_easy *data)
{
  struct curl_slist *list;
  Curl_share_lock(data, CURL_LOCK_DATA_COOKIE, CURL_LOCK_ACCESS_SINGLE);
  list = cookie_list(data);
  Curl_share_unlock(data, CURL_LOCK_DATA_COOKIE);
  return list;
}

void Curl_flush_cookies(struct Curl_easy *data, bool cleanup)
{
  Curl_share_lock(data, CURL_LOCK_DATA_COOKIE, CURL_LOCK_ACCESS_SINGLE);
  /* only save the cookie file if a transfer was started (cookies->running is
     set), as otherwise the cookies were not completely initialized and there
     might be cookie files that were not loaded so saving the file is the
     wrong thing. */
  if(data->cookies) {
    if(data->set.str[STRING_COOKIEJAR] && data->cookies->running) {
      /* if we have a destination file for all the cookies to get dumped to */
      CURLcode result = cookie_output(data, data->cookies,
                                      data->set.str[STRING_COOKIEJAR]);
      if(result)
        infof(data, "WARNING: failed to save cookies in %s: %s",
              data->set.str[STRING_COOKIEJAR], curl_easy_strerror(result));
    }

    if(cleanup && (!data->share || (data->cookies != data->share->cookies))) {
      Curl_cookie_cleanup(data->cookies);
      data->cookies = NULL;
    }
  }
  Curl_share_unlock(data, CURL_LOCK_DATA_COOKIE);
}

void Curl_cookie_run(struct Curl_easy *data)
{
  Curl_share_lock(data, CURL_LOCK_DATA_COOKIE, CURL_LOCK_ACCESS_SINGLE);
  if(data->cookies)
    data->cookies->running = TRUE;
  Curl_share_unlock(data, CURL_LOCK_DATA_COOKIE);
}

#endif /* CURL_DISABLE_HTTP || CURL_DISABLE_COOKIES */
